from fastapi import APIRouter, Depends, HTTPException, Body
from app.db import database
from app.v1.models.platform.tasks import Task, TaskCreate, TaskUpdate, TaskList
from app.v1.models.platform.services import CheckListSelect
from app.v1.models.saas.usersmodel import User
from app.v1.dependencies.auth import get_current_userdetails
from app.v1.services.platform import tasks as task_service
from app.v1.services.platform.services import get_select_list
from typing import Optional
from datetime import datetime

router = APIRouter()

@router.post("/", response_model=Task)
def create_task(task: TaskCreate, db: database.MongoDB = Depends(database.get_mongo_db)):
    return task_service.create_task_service(task, db)

@router.get("/getchecklist/{account_id}", response_model=CheckListSelect)
def get_checklist(account_id: str, db: database.MongoDB = Depends(database.get_mongo_db), current_user: User = Depends(get_current_userdetails)):
    return get_select_list(account_id, db, current_user)

@router.get("/", response_model=TaskList)
def list_all_tasks(account_id: str, service_id: Optional[str] = None, task_mode: Optional[str] = None, priority: Optional[str] = None, roundtrip: Optional[bool] = None, from_date: Optional[str] = None, to_date: Optional[str] = None, customer_id: Optional[str] = None, workforce_id: Optional[str] = None, fleet_id: Optional[str] = None, status: Optional[str] = None, q: Optional[str] = None, skip: int = 0, limit: int = 10, sort_by: Optional[str] = None,   sort_order: Optional[str] = "asc", db: database.MongoDB = Depends(database.get_mongo_db)):
    print("list_all_taskslist_all_taskslist_all_tasks222222222222")
    return task_service.list_tasks_service(account_id, service_id, task_mode, priority, roundtrip, from_date, to_date, customer_id, workforce_id, fleet_id, status, q, skip, limit, sort_by, sort_order, db)

@router.get("/summary")
def get_task_summary(
    customer_id: Optional[str] = None,
    workforce_id: Optional[str] = None,
    fleet_id: Optional[str] = None,
    db: database.MongoDB = Depends(database.get_mongo_db)
):
    try:
        payload = {
            "customer_id": customer_id,
            "workforce_id": workforce_id,
            "fleet_id": fleet_id
        }
        return task_service.get_task_summary_service(payload, db)
    except ValueError as e:
        raise HTTPException(status_code=400, detail=str(e))

@router.get("/{task_id}", response_model=Task)
def get_task(task_id: str, db: database.MongoDB = Depends(database.get_mongo_db)):
    task = task_service.get_task_service(task_id, db)
    if not task:
        raise HTTPException(status_code=404, detail="Task not found")
    return task

@router.post("/{task_id}", response_model=Task)
def update_task(task_id: str, update: TaskUpdate, db: database.MongoDB = Depends(database.get_mongo_db)):
    task = task_service.update_task_service(task_id, update, db)
    if not task:
        raise HTTPException(status_code=404, detail="Task not found")
    return task

@router.delete("/{task_id}", response_model=Task)
def delete_task(task_id: str, db: database.MongoDB = Depends(database.get_mongo_db)):
    task = task_service.delete_task_service(task_id, db)
    if not task:
        raise HTTPException(status_code=404, detail="Task not found")
    return task

@router.get("/list/{account_id}/", response_model=TaskList)
def list_tasks(account_id: str, db: database.MongoDB = Depends(database.get_mongo_db), current_user: User = Depends(get_current_userdetails)):
    return task_service.list_tasks_service(account_id, db)

# TaskBar -- > Total Tasks, Completed, Cancelled Details
# @router.post("/tasksummary/")
# def get_task_summary(
#     payload: dict = Body(...),
#     db: database.MongoDB = Depends(database.get_mongo_db),
#     current_user: User = Depends(get_current_userdetails)
# ):
#     return task_service.get_task_summary_service(payload, db)


